#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
from typing import Callable, Any, TypeAlias, TypeVar, Generic

from .create_function import create_function


_T = TypeVar('_T', bound=str)


class JsonPath(list[_T], Generic[_T]):
    pass


JsonPathStr: TypeAlias = JsonPath[str]


AttrStructure: TypeAlias = (
    JsonPathStr |
    list['AttrStructure'] |
    tuple['AttrStructure', ...] |
    dict[str, 'AttrStructure']
)


def make_parse_func(structure: AttrStructure) -> Callable[[Any], Any]:
    inner_code, _ = _inner(structure, "payload", 0)
    fn = create_function("parse", ["payload"], [f"return {inner_code}"])
    return fn


def _inner(structure: AttrStructure, outer: str, depth: int) -> tuple[str, list[str]]:
    if isinstance(structure, JsonPath):
        return _produce_attr_code(structure, outer, depth)
    if isinstance(structure, dict):
        return _produce_dict_code(structure, outer, depth)
    if isinstance(structure, list):
        return _produce_list_code(structure, outer, depth)
    if isinstance(structure, tuple):
        return _produce_tuple_code(structure, outer, depth)
    raise TypeError


def _produce_dict_code(structure: dict, outer: str, depth: int) -> tuple[str, list[str]]:
    lines: list[str] = []
    common_path = None
    for dict_key, sub_structure in structure.items():
        right_value, path = _inner(sub_structure, outer, depth)
        if common_path is None:
            common_path = path
        else:
            path = path[:len(common_path)]
            assert path == common_path, f"{path=} != {common_path=}"
        lines.append(f'{repr(dict_key)}: {right_value}')
    body = f", \n{' ' * (depth+1)}".join(lines)
    if common_path is None:
        raise ValueError
    result = f"""{' ' * depth}{{
{' ' * (depth+1)}{body}
{' ' * depth}}}
    """
    return result, common_path


def _produce_tuple_code(structure: tuple, outer: str, depth: int) -> tuple[str, list[str]]:
    lines: list[str] = []
    common_path = None
    for sub_structure in structure:
        inner_value, path = _inner(sub_structure, outer, depth)
        if common_path is None:
            common_path = path
        else:
            path = path[:len(common_path)]
            assert path == common_path, f"{path=} != {common_path=}"
        lines.append(f'{inner_value}')
    body = f", \n{' ' * (depth+1)}".join(lines)
    if common_path is None:
        raise ValueError
    result = f"""{' ' * depth}(
{' ' * (depth+1)}{body}
{' ' * depth})
    """
    return result, common_path


def _produce_list_code(structure: list, outer: str, depth: int) -> tuple[str, list[str]]:
    assert len(structure) == 1
    sub_structure = structure[0]
    i_name = f'_{depth+1}'
    value, path = _inner(sub_structure, outer=i_name, depth=depth + 1)
    key = path[depth]
    result = f"""{' ' * depth}[
{' ' * (depth + 1)}{value}
{' ' * (depth + 1)}for {i_name} in {outer}[{repr(key)}]
{' ' * depth}] 
    """
    return result, path


def _produce_attr_code(structure: JsonPathStr, outer: str, depth: int) -> tuple[str, list[str]]:
    assert len(structure) == depth + 1, f"Depth and path mismatch: {structure}, {depth}"
    return f"{outer}[{repr(structure[-1])}]", structure[:-1]

