#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
#
import asyncio
import itertools
from logging import getLogger

logger = getLogger(__name__)


class AsyncInitable:
    def __init__(self):
        self.async_init_scheduled__event: asyncio.Event = asyncio.Event()  # called async_init
        self._tasks_before_async_init: list[asyncio.Task] = []
        self.async_init_started__event: asyncio.Event = asyncio.Event()  # passed wait
        self.async_init_finished__event: asyncio.Event = asyncio.Event()  # done own _async_init

    def wait_before_async_init_for(self, task: asyncio.Task) -> None:
        self._tasks_before_async_init.append(task)

    async def async_init(self) -> None:
        if self.async_init_scheduled__event.is_set():
            logger.warning(f"async_start called extra time on {self}")
            return
        self.async_init_scheduled__event.set()
        if self._tasks_before_async_init:
            logger.debug(f"{self} before starting own _async_init is waiting for {self._tasks_before_async_init}")
            await asyncio.gather(*self._tasks_before_async_init)
        self.async_init_started__event.set()
        logger.debug(f"{self} started _async_init")
        await self._async_init()
        logger.debug(f"{self} finished _async_init")
        self.async_init_finished__event.set()

    async def _async_init(self) -> None:
        await self._async_init_attrs()

    async def _async_init_attrs(self) -> None:
        async with asyncio.TaskGroup() as tg:
            for attribute_name in vars(self):
                attribute = getattr(self, attribute_name)
                if isinstance(attribute, AsyncInitable) and not attribute.async_init_scheduled__event.is_set():
                    tg.create_task(attribute.async_init())
                if isinstance(attribute, (list, set)):
                    for val in attribute:
                        if isinstance(val, AsyncInitable) and not val.async_init_scheduled__event.is_set():
                            tg.create_task(val.async_init())
                if isinstance(attribute, dict):
                    for val in itertools.chain(attribute.keys(), attribute.values()):
                        if isinstance(val, AsyncInitable) and not val.async_init_scheduled__event.is_set():
                            tg.create_task(val.async_init())
