import itertools
from dataclasses import dataclass, InitVar, field
from typing import Union, List, Any, Callable, Hashable

from .default_caster import CastType
from .extras import NoDefault


@dataclass
class Item:
    source: Union[Hashable, list['Item']]
    cast_as: Union[None, CastType, List[CastType]]
    rename: InitVar[Hashable] = None
    optional: bool = False
    default: Any = NoDefault
    preprocessor: Callable[[Any], Any] = None
    postprocessor: Callable[[Any], Any] = None
    result_key: Hashable = field(init=False)

    def __post_init__(self, rename: Hashable):
        if not isinstance(self.source, Hashable) and not rename:
            raise TypeError(f'Item with non hashable key requires "rename"')
        self.result_key = self.source if rename is None else rename

    def preprocess(self, value: Any) -> Any:
        return self.preprocessor(value) if self.preprocessor else value

    def postprocess(self, value: Any) -> Any:
        return self.postprocessor(value) if self.postprocessor else value

    def get_keys(self) -> set[Hashable]:
        if isinstance(self.source, Hashable):
            return {self.source}
        else:
            return set(itertools.chain(item.source for item in self.source))
