#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
#
from typing import Any, Optional, Callable, Hashable

from . import NoDefault
from .item import Item


class Grouping:
    def __init__(self, item: Item, *items: Item):
        self.items = [item] + list(items)

    def reinit(self):
        pass

    def on_item_got(self, item: Item, value: Any) -> None:
        pass

    def finish(self):
        pass

    def __str__(self):
        cls = type(self)
        return f"{cls.__name__}({','.join(repr(item.source) for item in self.items)})"


class OneOf(Grouping):
    def __init__(self, item: Item, *items: Item):
        super().__init__(item, *items)
        for item in self.items:
            item.optional = True
        self.missing_items_amount: int = 0
        self.founded_item: Optional[Item] = None

    def reinit(self):
        self.missing_items_amount = 0
        self.founded_item = None

    def on_item_got(self, item: Item, value: Any) -> None:
        if value is NoDefault:
            self.on_missing_item()
        else:
            self.on_found_item(item)

    def on_missing_item(self):
        self.missing_items_amount += 1
        if self.missing_items_amount == len(self.items):
            raise ValueError(f"required exactly one of: {[item.source for item in self.items]}, found nothing")

    def on_found_item(self, item: Item):
        if self.founded_item:
            raise ValueError(f"required exactly one of: {[item.source for item in self.items]}, "
                             f"got {self.founded_item.source} and {item.source}")
        self.founded_item = item

    def finish(self):
        if self.founded_item is None:
            raise ValueError(f"required exactly one of: {[item.source for item in self.items]}, found nothing")
