#  Copyright (C) 2022
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Ilya Leontyev <i.leontyev@abm-jsc.ru>

import logging
from dataclasses import dataclass, InitVar, field
from enum import IntEnum
from typing import Optional

from http_tools.mime_types import ContentType

from http_tools.http_server_connector import HttpServerConnector


logger = logging.getLogger(__name__)


class SendEmailError(Exception):
    pass


class GetEmailError(Exception):
    pass


class SendPriority(IntEnum):
    high = 1
    medium = 2


@dataclass
class AttachedFile:
    filename: str
    content: bytes
    content_type: ContentType = ContentType.Octet.value


@dataclass
class Email:
    subject: str
    body: str
    send_to: InitVar[list[str]] = None
    copy_to: InitVar[list[str]] = None
    hidden_copy_to: InitVar[list[str]] = None
    to_recipients: list[str] = field(init=False)
    cc_recipients: list[str] = field(init=False)
    bcc_recipients: list[str] = field(init=False)

    def __post_init__(self,
                      send_to: Optional[list[str]],
                      copy_to: Optional[list[str]],
                      hidden_copy_to: Optional[list[str]]) -> None:
        self.to_recipients = send_to or []
        self.cc_recipients = copy_to or []
        self.bcc_recipients = hidden_copy_to or []


@dataclass
class SendSettings:
    sender_name: str
    send_priority: SendPriority


@dataclass
class SendingResult:
    done: bool
    result: int


@dataclass
class GettingResult:
    done: bool
    result: dict[str, str]


class EmailServiceV2Connector:
    @dataclass
    class Config(HttpServerConnector.Config):
        server_name: str = ''
        sender_name: str = ''
        send_priority: SendPriority = SendPriority.medium

    Context = HttpServerConnector.Context

    def __init__(self, config: Config, context: Context):
        self._config = config
        self._context = context
        self._connector = HttpServerConnector(config, context)

    async def get_email(self, email_id: int) -> GettingResult:
        answer = await self._connector.get(
            path="/v2/email/get", url_query={"email_id": email_id}, headers={"server_name": self._config.server_name},
        )
        try:
            getting_result = GettingResult(**answer)
        except TypeError as e:
            error_message = f"Failed to parse answer from EmailServer. Error: {e}"
            logger.error(error_message)
            raise GetEmailError(error_message) from e

        if not getting_result.done:
            error_message = f"Failed to get email message. Negative answer from EmailServer"
            logger.error(error_message)
            raise GetEmailError(error_message)

        return getting_result

    async def send_email(self,
                         email: Email,
                         send_priority: Optional[SendPriority] = None) -> SendingResult:
        logger.debug("send_email: %s", email)

        if not (email.to_recipients or email.cc_recipients or email.bcc_recipients):
            raise ValueError("Got no recipients")

        payload = {
            "values": {
                "email": email,
                "send_settings": SendSettings(
                    self._config.sender_name,
                    send_priority if send_priority else self._config.send_priority,
                ),
            }
        }

        answer = await self._connector.post_json(
            path='/v2/email/send', payload=payload, headers={"server_name": self._config.server_name}
        )

        try:
            sending_result = SendingResult(**answer)
        except TypeError as err:
            error_message = f'Failed to parse answer from EmailServer. Error: {err}'
            logger.error(error_message)
            raise SendEmailError(error_message) from err

        if not sending_result.done:
            error_message = f'Failed to send email message. Negative answer from EmailServer'
            raise SendEmailError(error_message)

        return sending_result 
