#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
from dataclasses import dataclass

import typed_json.typed_In_dumps
from entity_read import sql
from entity_read.entity import Entity
from entity_read.sql.atoms import Selectable

from .expression import Expression
from .literal import Literal


key_to_function_factory = {
    "lt": sql.inter.FunctionLessThan,
    "le": sql.inter.FunctionLessEqual,
    "eq": sql.inter.FunctionEqual,
    "ne": sql.inter.FunctionNonEqual,
    "ge": sql.inter.FunctionGreaterEqual,
    "gt": sql.inter.FunctionGreater,
    "in": sql.inter.FunctionIn,
    "not_in": sql.inter.FunctionNotIn,
    "or": sql.inter.FunctionOr,
    "and": sql.inter.FunctionAnd,
    "not": sql.inter.FunctionNot,
    "like": lambda args: sql.inter.FunctionStringContains(
        args=(sql.inter.FunctionCastToStr(args=args[:1]), args[1]), ignore_case=False),
    "ilike": lambda args: sql.inter.FunctionStringContains(
        args=(sql.inter.FunctionCastToStr(args=args[:1]), args[1]), ignore_case=True),
    "is_null": sql.inter.FunctionIsNull,
    "is_not_null": sql.inter.FunctionIsNotNull,
}


@dataclass(frozen=False)
class Condition(Expression):
    key: str
    args: list[Expression]

    def __repr_in_dumps__(self) -> str | dict:
        self_dict_repr: dict = super().__repr_in_dumps__()
        arg_reprs = [arg.__repr_in_dumps__() for arg in self_dict_repr["args"]]
        if all(isinstance(arg_repr, str) for arg_repr in arg_reprs):
            return f'cond.{self.key}({",".join(arg_reprs)})'
        self_dict_repr['key'] = f"'{self_dict_repr['key']}'"
        self_dict_repr['args'] = arg_reprs
        self_dict_repr[typed_json.typed_In_dumps.TYPE_KEY] = self.get_prefix() + Condition.__name__
        return self_dict_repr

    def eval(self, entity_type: type[Entity], variables: dict[str, Selectable]) -> sql.atoms.function.Function:
        if (function_factory := key_to_function_factory.get(self.key)) is None:
            raise KeyError(f"Not found condition {self.key!r}")
        # noinspection PyArgumentList
        return function_factory(args=[arg.eval(entity_type, variables) for arg in self.args])


class ConditionEq(Condition):
    def __init__(self, left: Expression, right: Expression):
        super().__init__(key="eq", args=[left, right])


class ConditionNe(Condition):
    def __init__(self, left: Expression, right: Expression):
        super().__init__(key="ne", args=[left, right])


class ConditionLt(Condition):
    def __init__(self, left: Expression, right: Expression):
        super().__init__(key="lt", args=[left, right])


class ConditionLe(Condition):
    def __init__(self, left: Expression, right: Expression):
        super().__init__(key="le", args=[left, right])


class ConditionGe(Condition):
    def __init__(self, left: Expression, right: Expression):
        super().__init__(key="ge", args=[left, right])


class ConditionGt(Condition):
    def __init__(self, left: Expression, right: Expression):
        super().__init__(key="gt", args=[left, right])


class ConditionIn(Condition):
    def __init__(self, left: Expression, right: Expression):
        super().__init__(key="in", args=[left, right])


class ConditionNotIn(Condition):
    def __init__(self, left: Expression, right: Expression):
        super().__init__(key="not_in", args=[left, right])


class ConditionIsNull(Condition):
    def __init__(self, left: Expression):
        super().__init__(key="is_null", args=[left, Literal(body=True)])


class ConditionIsNotNull(Condition):
    def __init__(self, left: Expression):
        super().__init__(key="is_null", args=[left, Literal(body=False)])


class ConditionOr(Condition):
    def __init__(self, *selectables: Expression):
        super().__init__(key="or", args=list(selectables))


class ConditionAnd(Condition):
    def __init__(self, *selectables: Expression):
        super().__init__(key="and", args=list(selectables))


class ConditionNot(Condition):
    def __init__(self, selectable: Expression):
        super().__init__(key="not", args=[selectable])
