#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
from dataclasses import dataclass, field

from entity_read import sql
from entity_read.entity import Entity

from ._eval_to_selector import eval_to_selector
from .expression import Expression
from .order import Order
from .query_token import QueryToken
from .remote_entity import RemoteEntity


@dataclass(frozen=True, repr=False)
class Query(QueryToken):
    entity: type[RemoteEntity | Entity] | None = None
    attrs: list[Expression] | Expression = field(default_factory=list)
    vars: dict[str, Expression] = field(default_factory=dict)
    filters: list[Expression] = field(default_factory=list)
    searches: list[Expression] = field(default_factory=list)
    orders: list[Order] = field(default_factory=list)
    limit: int | None = None
    offset: int | None = None
    is_scalar: bool | None = None

    def __post_init__(self):
        if self.is_scalar is None:
            object.__setattr__(self, 'is_scalar', not isinstance(self.attrs, list))

    def __repr_in_dumps__(self) -> dict:
        result = super().__repr_in_dumps__()
        del result["entity"]
        return result

    def eval(self, entity_type: type[Entity]) -> sql.lower.LowerSelector:
        return eval_to_selector(
            entity_type=entity_type, local_column_to_remote={},
            attrs=self.attrs, vars_=self.vars, filters=self.filters, searches=self.searches,
            orders=self.orders, limit=self.limit, offset=self.offset, is_scalar=self.is_scalar
        )
