#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
import abc
import itertools
from dataclasses import dataclass, field, fields
from typing import Union, Any

from ..context import SqlContext
from ..node import Node, SQL
from .abstract import Selectable, BooleanSelectable
from .column import EColumn
from .literal import ELiteral


@dataclass
class EFunction2(Selectable, abc.ABC):
    _name = None

    def get_name(self) -> str:
        return f'{self._name}(NotImplemented)'

    def _items(self) -> list[tuple[str, Any]]:
        return [(field_.name, val) for field_ in fields(self) if (val := getattr(self, field_.name))]

    def requires(self) -> list[EColumn]:
        return list(itertools.chain(*[value.requires() for key, value in self._items() if isinstance(value, Node)]))

    def to_sql(self, context: SqlContext, key_to_expression=None) -> SQL:
        return self._to_sql(**{
            key: value.to_sql(context, key_to_expression) for key, value in self._items() if isinstance(value, Node)})

    @abc.abstractmethod
    def _to_sql(self, **kwargs: dict[str, SQL]) -> SQL:
        ...


@dataclass
class EBooleanFunction2(EFunction2, BooleanSelectable, abc.ABC):
    pass


@dataclass
class EFunctionEqual(EBooleanFunction2):
    _name = "Equal"
    left: Selectable
    right: Selectable

    def _to_sql(self, left: str, right: str) -> SQL:
        return f'{left} = {right}'

# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #


class EBoolean:
    pass


@dataclass
class EFunction(Selectable, abc.ABC):
    args: list[Union[EColumn, 'EFunction', ELiteral]] = field(default_factory=list)
    kwargs: dict[str, Union[EColumn, 'EFunction', ELiteral]] = field(default_factory=dict)

    def requires(self) -> list[EColumn]:
        required = []
        for arg in itertools.chain(self.args, self.kwargs.values()):
            if isinstance(arg, Node):
                required += arg.requires()
        return required
    
    def get_name(self) -> str:
        pass

    def to_sql(self, context: SqlContext, key_to_expression: dict = None) -> str:
        args_sql: list[str] = [arg.to_sql(context, key_to_expression) for arg in self.args]
        kwargs_sql: dict[str, str] = {k: v.to_sql(context, key_to_expression) for k, v in self.kwargs.items()}
        return self._to_sql(args_sql, kwargs_sql)

    @abc.abstractmethod
    def _to_sql(self, args_sql: list, kwargs_sql: dict) -> str:
        ...


@dataclass
class EBooleanFunction(EFunction, EBoolean, abc.ABC):
    pass


@dataclass
class EFunctionLE(EBooleanFunction):
    def _to_sql(self, args_sql: list, kwargs_sql: dict) -> str:
        return f'{args_sql[0]} <= {args_sql[1]}'


@dataclass
class EFunctionEQ(EBooleanFunction):
    def _to_sql(self, args_sql: list, kwargs_sql: dict) -> str:
        return f'{args_sql[0]} = {args_sql[1]}'


@dataclass
class EFunctionNE(EBooleanFunction):
    def _to_sql(self, args_sql: list, kwargs_sql: dict) -> str:
        return f'{args_sql[0]} != {args_sql[1]}'


@dataclass
class EFunctionGE(EBooleanFunction):
    def _to_sql(self, args_sql: list, kwargs_sql: dict) -> str:
        return f'{args_sql[0]} >= {args_sql[1]}'


@dataclass
class EFunctionGT(EBooleanFunction):
    def _to_sql(self, args_sql: list, kwargs_sql: dict) -> str:
        return f'{args_sql[0]} > {args_sql[1]}'


@dataclass
class EFunctionLT(EBooleanFunction):
    def _to_sql(self, args_sql: list, kwargs_sql: dict) -> str:
        return f'{args_sql[0]} < {args_sql[1]}'


@dataclass
class EFunctionIncluded(EBooleanFunction):
    def _to_sql(self, args_sql: list, kwargs_sql: dict) -> str:
        if args_sql[1] != "'{}'":
            return f'{args_sql[0]} = ANY ({args_sql[1]})'
        return f'FALSE'

    def requires(self) -> list[EColumn]:
        if not self.args[1]:
            return []
        return super().requires()


@dataclass
class EFunctionStringContains(EBooleanFunction):
    def _to_sql(self, args_sql: list, kwargs_sql: dict) -> str:
        return f"{args_sql[0]} ilike {args_sql[1]}"


@dataclass
class EFunctionIsNull(EBooleanFunction):
    def _to_sql(self, args_sql: list, kwargs_sql: dict) -> str:
        return f"{args_sql[0]} IS NULL"


@dataclass
class EFunctionNotNull(EBooleanFunction):
    def _to_sql(self, args_sql: list, kwargs_sql: dict) -> str:
        return f"{args_sql[0]} IS NOT NULL"


@dataclass
class EFunctionOr(EBooleanFunction):
    def _to_sql(self, args_sql: list, kwargs_sql: dict) -> str:
        return f"({' OR '.join(args_sql)})"


@dataclass
class EFunctionAnd(EBooleanFunction):
    def _to_sql(self, args_sql: list, kwargs_sql: dict) -> str:
        return f"({' AND '.join(args_sql)})"


@dataclass
class EFunctionLen(EFunction):  # Integer
    def _to_sql(self, args_sql: list, kwargs_sql: dict) -> str:
        return f'jsonb_array_length({args_sql[0]})'


@dataclass
class EFunctionGetJsonField(EFunction):
    def _to_sql(self, args_sql: list, kwargs_sql: dict) -> str:
        return f"jsonb_path_query({args_sql[0]}, '$.{args_sql[1]}')"


@dataclass
class EFunctionGetJsonFieldFromEach(EFunction):
    def _to_sql(self, args_sql: list, kwargs_sql: dict) -> str:
        return f"jsonb_path_query_array({args_sql[0]}, '$.{args_sql[1]}')"


@dataclass
class EFunctionCastToStr(EFunction):
    def _to_sql(self, args_sql: list, kwargs_sql: dict) -> str:
        return f"{args_sql[0]}::TEXT"
