#  Copyright (C) 2022
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
#
import json
from dataclasses import dataclass, asdict
from typing import Union

import init_helpers
from async_tools import AsyncOnStop, AsyncOnStart
from dict_caster import DictCaster, Item
from init_helpers import dict_to_dataclass
from entities.organization import Organization
from sqlalchemy_tools.database_connector.database_connector import DatabaseConnector

from entity_view.viewable_entity import InsertView
from extended_logger import get_logger
from proposal import EntityDatabase
from template.delta_view_weblate_renderer import DeltaViewWeblateRenderer
from template.weblate_email_renderer import WeblateEmailRenderer
from controller import Controller
from entities.user_to_organization import UserToOrganization
from entities.access_inner_request import AccessInnerRequest
from translation_loader import TranslationLoader

logger = get_logger(__name__)


@dataclass
class Initer:
    @dataclass
    class Config:
        translation_loader: TranslationLoader.Config
        delta_view_renderer: DeltaViewWeblateRenderer.Config
        email_renderer: WeblateEmailRenderer.Config
        logging: init_helpers.LogsConfig
        controller: Controller.Config
        organization_database: DatabaseConnector.Config

    config: Config

    @dataclass
    class Context(AsyncOnStart, AsyncOnStop):
        company: str = "ABM"
        project_group: str = "SafeRegion"
        project_name: str = "OrganizationService"
        translation_loader: TranslationLoader = None
        delta_view_renderer: DeltaViewWeblateRenderer = None
        organization_database: DatabaseConnector = None
        entity_database: EntityDatabase = None
        controller: Controller = None

    context: Context

    def __init__(self):
        init_helpers.init_random()
        self.config = init_helpers.parse_args(config_file=init_helpers.Arg.ini_file_to_dataclass(self.Config))
        logger.info(f'init with config; {self.config}')
        init_helpers.init_logs(self.config.logging)
        self.context = self.Context()

    async def __aenter__(self) -> None:
        logger.info(f'start async initialisation')
        self.context.translation_loader = TranslationLoader(self.config.translation_loader)

        self.context.organization_database = DatabaseConnector(self.config.organization_database)
        self.context.entity_database = EntityDatabase(self.context)
        self.context.delta_view_renderer = DeltaViewWeblateRenderer(self.config.delta_view_renderer, self.context)
        await self.context.delta_view_renderer.load_templates_for_entity_types([
            Organization,
            UserToOrganization,
        ])

        self.context.controller = Controller(context=self.context, config=self.config.controller)
        await AsyncOnStart.start_if_necessary(self.context)
        logger.debug(f"controller: {self.context.controller}")
        logger.info(f"----===== Init done ====----")
        # self.created_contract()

        # await controller.test_type(Union[int, str])
        await self.created_org()

    async def created_org(self):
        organization = Organization(inn=123, ogrn=2, title='asd_rename', address_city='asd_rename', chat_id=1, created_at=1728557824.0,
                                    users=[
                                        UserToOrganization(organization_inn=123, user_id=123, created_at=1730207616.0, is_admin=False),
                                        UserToOrganization(organization_inn=123, user_id=124, created_at=1730207616.0, is_admin=False)
                                    ]
                                    )
        rendered_diff = await self.context.controller.render_diff_from_db(organization)
        # print(rendered_diff)
        # organization_to_insert = Organization(id = 2, ogrn=2, title='test', address_city='test', chat_id=1, created_at=1728557824.0,
        #                                       users=[UserToOrganization(organization_inn=222, user_id=123, created_at=1730207616.0, is_admin=True)]
        #                                       )
        organization_to_insert = Organization[InsertView](inn=222, ogrn=2, title='test', address_city='test', chat_id=1, created_at=1728557824.0,
                                              users=[
                                                  UserToOrganization[InsertView](organization_inn=222, user_id=124, created_at=1730207616.0, is_admin=True)
                                                     ]
                                              )
        # org_dict = asdict(organization_to_insert)
        # print(org_dict)
        # org_dict['users'] = asdict(UserToOrganization(organization_inn=2, user_id=123, created_at=1730207616.0, is_admin=True))
        # print(org_dict)
        # org_dict = {"org": org_dict}
        # print(f"{org_dict=}")
        # organization_to_insert = DictCaster(Item("org", Organization[InsertView])).cast_and_return(org_dict)
        # organization_to_insert = dict_to_dataclass(org_dict, Organization[InsertView])
        # organization_to_insert = dict_to_dataclass(org_dict, Organization)
        # print(organization_to_insert)
        rendered_entity = self.context.delta_view_renderer.render(organization_to_insert)
        print(f"{rendered_entity=}")
        # await self.context.controller.insert(organization_to_insert)

    async def created_request(self):
        organization = AccessInnerRequest(id=123, chat_id=2, title='asd_rename', address_city='asd_rename', created_at=1728557824.0)
        diff = await self.context.controller.render_diff_from_db(organization)
        print(diff)

    async def __aexit__(self, exc_type, exc_val, exc_tb):
        await AsyncOnStop.stop_if_necessary(self.context)
        logger.info(f"----===== Deinit done ====----")
