#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

import logging
from dataclasses import dataclass

from async_tools import AsyncOnStart
from dict_caster.extras import to_list

from ..abstract_file_storage import AbstractFileStorage, FileSource, FileDeleteResult, FileDeleteStatus
from ..exceptions import BucketAlreadyExist
from ..s3_file_storage.s3_interface import S3Interface, Range

logger = logging.getLogger(__name__)


class S3FileStorage(AbstractFileStorage, AsyncOnStart):
    @dataclass
    class Config(S3Interface.Config):
        bucket_name: str = "abm-storage"

    def __init__(self, config: Config):
        logger.info(f"{type(self).__name__} init. Config: {config}")
        self._s3_interface = S3Interface(config)
        self._bucket_name = config.bucket_name

    async def _on_start(self) -> None:
        try:
            await self._s3_interface.create_bucket(self._bucket_name)
        except BucketAlreadyExist:
            logger.info(f"Bucket already exist. bucket_name = {self._bucket_name}")
        else:
            logger.info(f"Bucket created. bucket_name = {self._bucket_name}")

    async def save(self, content: bytes, relative_path: str, name: str, allow_rewrite: bool = False) -> None:
        await self._s3_interface.upload_file(
            bucket_name=self._bucket_name, payload=content, key=f"{relative_path}/{name}", allow_rewrite=allow_rewrite)

    async def load(self, relative_path: str, name: str, offset: int = 0, size: int = -1) -> bytes:
        return await self._s3_interface.get_file(
            bucket_name=self._bucket_name, key=f"{relative_path}/{name}", range_=Range(offset, size)
        )

    async def delete(self, file_sources: list[FileSource] | FileSource) -> list[FileDeleteResult]:
        file_sources = to_list(file_sources)
        if not file_sources:
            return []

        delete_answer = await self._s3_interface.delete_files(
            bucket_name=self._bucket_name,
            keys=[f"{file_source.relative_path}/{file_source.name}" for file_source in file_sources],
        )
        file_delete_results = []
        for file in delete_answer.Deleted:
            file_delete_results.append(
                FileDeleteResult(
                    file_source=FileSource(
                        relative_path=self._cast_relative_path_from_file_full_path(file.Key),
                        name=self._cast_file_name_from_full_path(file.Key),
                    ),
                    status=FileDeleteStatus.OK,
                )
            )
        for file in delete_answer.Errors:
            file_delete_results.append(
                FileDeleteResult(
                    file_source=FileSource(
                        relative_path=self._cast_relative_path_from_file_full_path(file.Key),
                        name=self._cast_file_name_from_full_path(file.Key),
                    ),
                    status=FileDeleteStatus.FAILED,
                    reason=file.Code,
                )
            )
        return file_delete_results

    @staticmethod
    def _cast_relative_path_from_file_full_path(full_path: str) -> str:
        return "/".join(full_path.split("/")[:-1])

    @staticmethod
    def _cast_file_name_from_full_path(full_path: str) -> str:
        return full_path.split("/")[-1]

    async def check_file_existence(self, relative_path: str, name: str) -> bool:
        return await self._s3_interface.check_file_existence(
            bucket_name=self._bucket_name, key=f"{relative_path}/{name}")
