#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

import logging
import os
from dataclasses import dataclass
from pathlib import Path

from aiofile import AIOFile, Writer

from ..abstract_file_storage import AbstractFileStorage
from ..exceptions import FileAlreadyExists, PathDoesNotExists, InsecurePath

logger = logging.getLogger(__name__)


class DiscFileStorage(AbstractFileStorage):
    @dataclass
    class Config:
        location: str

    def __init__(self, config: Config):
        logger.info(f"{type(self).__name__} init. Location: {config.location}")
        self.location = config.location

    async def save(self, content: bytes, relative_path: str, name: str, allow_rewrite: bool = False) -> None:
        dir_path = f"{self.location}/{relative_path}"
        if not os.path.exists(dir_path):
            logger.info(f"Path = {dir_path} does not exists. Will be created")
            os.makedirs(dir_path)

        full_path = f"{dir_path}/{name}"
        logger.info(f"Full path = {full_path}")
        if Path(full_path).exists() and not allow_rewrite:
            raise FileAlreadyExists(full_path)

        logger.info(f"Saving file along the path: {full_path}")
        async with AIOFile(full_path, 'wb') as file:
            writer = Writer(file)
            await writer(content)

    async def load(self, relative_path: str, name: str, offset: int = 0, size: int = -1) -> bytes:
        dir_path = f"{self.location}/{relative_path}"
        full_path = f"{dir_path}/{name}"
        if not Path(full_path).exists():
            raise PathDoesNotExists(full_path)

        if not os.path.realpath(full_path).startswith(os.path.realpath(dir_path)):
            raise InsecurePath(full_path)

        logger.info(f"Loading file along the path: {full_path}")
        async with AIOFile(full_path, 'rb') as file:
            content = await file.read(size, offset)

        return content

    async def delete(self, relative_path: str, name: str) -> None:
        dir_path = f"{self.location}/{relative_path}"
        full_path = f"{dir_path}/{name}"

        if not Path(full_path).exists():
            raise PathDoesNotExists(full_path)

        logger.info(f"Deleting file along the path: {full_path}")
        os.remove(full_path)
        if not os.listdir(dir_path):
            os.rmdir(dir_path)

    async def check_file_existence(self, relative_path: str, name: str) -> bool:
        full_path = f"{self.location}/{relative_path}/{name}"
        return Path(full_path).exists()
