#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

from abc import ABC, abstractmethod
from dataclasses import dataclass, field
from typing import Any, Literal, Awaitable

from prometheus_tools.metrics_registry import MetricsRegistry

from file_storage.exceptions import FileStorageFull


@dataclass
class AbstractStorageStatus(ABC):
    total_b: float
    used_b: float
    free_b: float
    used_percentage: float = field(init=False)

    def __post_init__(self):
        self.used_percentage = self.calculate_used_percentage()

    def increase_used_space(self, content_size: int) -> None:
        self.used_b += content_size
        self.free_b -= content_size
        self.used_percentage = self.calculate_used_percentage()

    def decrease_used_space(self, content_size: int) -> None:
        self.used_b -= content_size
        self.free_b += content_size
        self.used_percentage = self.calculate_used_percentage()

    def calculate_used_percentage(self) -> float:
        return min(round(self.used_b/self.total_b * 100, 2), 100.0)

    def check_free_space(self, content_size: int) -> None:
        if content_size > self.free_b:
            raise FileStorageFull(f'Not enough disk space. Free storage size {self.free_b} bytes, '
                                  f'but need {content_size} bytes')

class AbstractFileStorage(ABC):
    @dataclass
    class Config:
        root_dir: str = ""
        max_capacity_gb: float = field(kw_only=True)

        def __post_init__(self) -> None:
            self.root_dir = self.root_dir.lstrip("/")

        @property
        def max_capacity_b(self) -> float:
            return self.max_capacity_gb * (1024 ** 3)

        @property
        def max_capacity_kb(self) -> float:
            return self.max_capacity_gb * (1024 ** 2)

        @property
        def max_capacity_mb(self) -> float:
            return self.max_capacity_gb * 1024

    @dataclass
    class Context:
        metrics: MetricsRegistry

    def __init__(self, config: Config, context: Context) -> None:
        self.config = config
        self.context = context
        self.class_name = type(self).__name__
        self.storage_status: AbstractStorageStatus | None = None

    async def save(self, key: str, value: bytes, allow_rewrite: bool = False) -> None:
        return await self._wrap_execution('save', self._save(key, value, allow_rewrite))

    async def load(self, key: str, offset: int = 0, size: int = -1) -> bytes:
        return await self._wrap_execution('load', self._load(key, offset, size))

    async def delete(self, key: str) -> None:
        return await self._wrap_execution('delete', self._delete(key))

    async def check_file_existence(self, key: str) -> None:
        return await self._wrap_execution('check_file_existence', self._check_file_existence(key))

    async def _wrap_execution(
            self, action: Literal['save', 'load', 'delete', 'check_file_existence'], handler: Awaitable[Any]
    ) -> Any:
        with self.context.metrics.track(
                'file_storage__handlers', {'storage': self.class_name, 'action': action}, except_labels={'error': None}
        ) as tracker:
            try:
                return await handler
            except Exception as er:
                tracker.labels['error'] = type(er).__name__
                raise

    def _split_key(self, key: str, division_index: int = 3) -> tuple[str, str]:
        if len(key) <= division_index:
            raise ValueError(f"Impossible to split '{key}' because the size is less or equal than {division_index}")
        return key[:division_index], key[division_index:]

    @abstractmethod
    async def _save(self, key: str, value: bytes, allow_rewrite: bool = False) -> None:
        pass

    @abstractmethod
    async def _load(self, key: str, offset: int = 0, size: int = -1) -> bytes:
        pass

    @abstractmethod
    async def _delete(self, key: str) -> None:
        pass

    @abstractmethod
    async def _check_file_existence(self, key: str) -> None:
        pass

    @abstractmethod
    async def _get_storage_status(self) -> AbstractStorageStatus:
        pass
