import tarfile
from zipfile import ZipFile, ZipInfo
from io import BytesIO
from .filename_preparation import shorten_file_name_by_bytes_size


class CustomZipFile:
    max_file_name_system_limit_in_bytes: int = 150

    def tar_buffer(self, files_metadata, file_payloads) -> BytesIO:
        buffer = BytesIO()

        with tarfile.open(fileobj=buffer, mode='w') as tar:
            for metadata in files_metadata:
                file_payload = file_payloads[metadata.file_md5]

                # If file name contains '/' zipfile lib thinking that it's subdirectories,
                # splitting name by it and creating archive with subdirectories
                # e.g. 'foo/bar.pdf' - in archive will be created folder foo and inside it file bar.pdf
                if '/' in metadata.full_file_name:
                    metadata.full_file_name = metadata.full_file_name.replace('/', '-')

                file_name = shorten_file_name_by_bytes_size(
                    metadata.full_file_name, self.max_file_name_system_limit_in_bytes
                )

                tf = tarfile.TarInfo(file_name)
                tf.size = len(file_payload)
                tar.addfile(tf, BytesIO(file_payload))

        return buffer

    def zip_buffer(self, files_metadata, file_payloads) -> BytesIO:
        buffer = BytesIO()

        with ZipFile(buffer, 'w') as zip_archive:
            for metadata in files_metadata:
                file_payload = file_payloads[metadata.file_md5]

                # If file name contains '/' zipfile lib thinking that it's subdirectories,
                # splitting name by it and creating archive with subdirectories
                # e.g. 'foo/bar.pdf' - in archive will be created folder foo and inside it file bar.pdf
                if '/' in metadata.full_file_name:
                    metadata.full_file_name = metadata.full_file_name.replace('/', '-')

                file_name = shorten_file_name_by_bytes_size(
                    metadata.full_file_name, self.max_file_name_system_limit_in_bytes
                )

                file = ZipInfo(file_name)
                zip_archive.writestr(file, file_payload)

        return buffer
