#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors:
#  Andrey Vaydich <a.vaydich@abm-jsc.ru>

from pathlib import Path


def convert_file_name(file_name: str, max_required_bytes_size: int) -> str:
    return shorten_file_name_by_bytes_size(file_name_replace_slash(file_name), max_required_bytes_size)

# If file name contains '/' zipfile lib thinking that it's subdirectories,
# splitting name by it and creating archive with subdirectories
# e.g. 'foo/bar.pdf' - in archive will be created folder foo and inside it file bar.
def file_name_replace_slash(file_name: str) -> str:
    return file_name.replace('/', '-')

# Each OS has maximum file name length
# so we must shorten it to avoid error while reading (zip file in particular)
def shorten_file_name_by_bytes_size(file_name: str, max_required_byte_size: int) -> str:
    file_name_byte_size = len(file_name.encode())

    if file_name_byte_size <= max_required_byte_size:
        return file_name

    file_path = Path(file_name)
    file_name_suffix = ''

    for datum in reversed(file_path.suffixes):
        if len(f'{file_name_suffix}{datum}'.encode()) > max_required_byte_size / 2:
            break

        file_name_suffix = f'{datum}{file_name_suffix}'

    file_name_suffix_byte_size = len(file_name_suffix.encode())

    if max_required_byte_size < file_name_suffix_byte_size:
        raise ValueError(
            f'max_required_byte_size({max_required_byte_size}) < file_name_suffix_byte_size({file_name_suffix_byte_size})'
        )

    file_name_splitted_by_suffix = file_name.split(file_name_suffix)
    file_name_without_suffix = file_name_splitted_by_suffix[0]

    max_required_byte_size = max_required_byte_size - file_name_suffix_byte_size
    short_file_name_prefix = ''

    for char in file_name_without_suffix:
        char_byte_size = len(char.encode())
        current_short_file_name_byte_size = len(short_file_name_prefix.encode())

        if current_short_file_name_byte_size + char_byte_size > max_required_byte_size:
            break

        short_file_name_prefix += char

    return f'{short_file_name_prefix}{file_name_suffix}'
