#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors:
#  Andrey Vaydich <a.vaydich@abm-jsc.ru>

from pathlib import Path


def convert_file_name(file_name: str, max_required_bytes_size: int) -> str:
    return shorten_file_name_by_bytes_size(file_name_replace_slash(file_name), max_required_bytes_size)

# If file name contains '/' zipfile lib thinking that it's subdirectories,
# splitting name by it and creating archive with subdirectories
# e.g. 'foo/bar.pdf' - in archive will be created folder foo and inside it file bar.
def file_name_replace_slash(file_name: str) -> str:
    return file_name.replace('/', '-')

# Each OS has maximum file name length
# so we must shorten it to avoid error while reading (zip file in particular)
def shorten_file_name_by_bytes_size(file_name: str, max_required_bytes_size: int) -> str:
    file_name_bytes_size = len(file_name.encode())

    if file_name_bytes_size <= max_required_bytes_size:
        return file_name

    file_name = Path(file_name)
    extension = file_name.suffix or ''
    extension_bytes_size = len(extension.encode())

    if max_required_bytes_size < extension_bytes_size:
        raise ValueError(f'required_bytes_size({max_required_bytes_size}) < extension_bytes_size({extension_bytes_size})')

    max_required_bytes_size = max_required_bytes_size - extension_bytes_size
    short_file_name = ''

    for symbol in file_name.stem:
        result_bytes_size = len(short_file_name.encode())
        symbol_bytes_size = len(symbol.encode())

        if result_bytes_size + symbol_bytes_size > max_required_bytes_size:
            break

        short_file_name += symbol

    short_file_name += extension

    return short_file_name
