#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors:
#  Vasya Svintsov <v.svintsov@techokert.ru>,
#  Alexander Medvedev <a.medvedev@abm-jsc.ru>,
#  Andrey Vaydich <a.vaydich@abm-jsc.ru>
import logging
import typing
from dataclasses import dataclass

from file_storage.abstract_file_storage import AbstractFileStorage, StorageCapacity
from file_storage.exceptions import FileAlreadyExists, PathDoesNotExists
from sqlalchemy import select, func

from .extra import AnySession
from .entities.file_info import FileInfo
from .database import Database, WrappedSession
from .tools.key_lock import KeyLock

logger = logging.getLogger(__name__)
FI = typing.TypeVar('FI', bound=FileInfo)


class FileKeeper(typing.Generic[FI]):
    @dataclass
    class Config:
        pass

    @dataclass
    class Context:
        database: Database
        file_storage: AbstractFileStorage

    def __init__(self, context: Context, config: Config = None, file_class: type[FI] = FileInfo) -> None:
        self.config = config or self.Config()
        self.context = context
        self.file_class = file_class
        self._file_lock = KeyLock()

    async def _save_to_file_storage(self, content: bytes, key: str) -> None:
        try:
            await self.context.file_storage.save(key, content)
        except FileAlreadyExists as e:
            if await self._load_from_file_storage(key) == content:
                logger.warning("File already exists, but matches current. Possible index lost")
            else:
                raise
        logger.info(f"{self.context.file_storage.status} after upload, used % {self.context.file_storage.status.used_percentage}")

    async def _load_from_file_storage(self, key: str, byte_range: range | None = None) -> bytes:
        byte_range = range(0, -1) if byte_range is None else byte_range
        return await self.context.file_storage.load(key, byte_range.start, byte_range.stop)

    async def _delete_from_file_storage(self, key: str) -> None:
        try:
            await self.context.file_storage.delete(key)
        except PathDoesNotExists as e:
            logger.warning("File to delete does not exist in storage. Possible index lost")
        logger.info(f"{self.context.file_storage.status} after deletion, used % {self.context.file_storage.status.used_percentage}")

    async def _set_file_storage_capacity(self) -> None:
        capacity = self.context.file_storage.config.max_capacity_b
        used = await self._get_total_size() or 0.0
        self.context.file_storage.status = StorageCapacity(max_b=capacity, used_b=used)
        logger.info(f"{self.context.file_storage.status}, used % {self.context.file_storage.status.used_percentage}")

    async def _get_bytes_info(self, key: str, session: WrappedSession, nullable: bool = False) -> FI | None:
        # noinspection PyTypeChecker
        query = select(self.file_class).where(self.file_class.key == key)
        return await (session.scalar_or_none(query) if nullable else session.scalar(query))

    async def _get_bytes_infos(self, keys: list[str], session: WrappedSession) -> list[FI]:
        # noinspection PyUnresolvedReferences
        return await session.scalars(select(self.file_class).where(self.file_class.key.in_(keys)))

    async def _get_total_size(self, session: AnySession = None) -> int | None:
        # noinspection PyTypeChecker
        async with self.context.database.ensure_session(session) as session:
            return await session.scalar_or_none(select(func.sum(self.file_class.size)))

    async def upload(self, content: bytes, session: AnySession = None, file_kwargs: dict | None = None) -> FI:
        logger.info(f'Uploading file, size: {len(content)} bytes')
        new_info: FI = self.file_class.from_bytes(content, **(file_kwargs or {}))
        new_key = new_info.key
        async with self._file_lock.restrict(new_key), self.context.database.ensure_session(session) as session:
            session: WrappedSession
            if present_info := await self._get_bytes_info(new_key, session, nullable=True):
                logger.info(f'Attempt to upload file identical to present with key: {new_key}')
                assert present_info == new_info, f"{new_info=} has same key as {present_info=}, but differs"
            else:
                logger.info(f'Not found saved file with key {new_key}. It will be uploaded')
                await self._save_to_file_storage(content, new_key)
                await session.add_and_flush(new_info)
        return new_info

    async def get(self, key: str, byte_range: range | None = None, session: AnySession = None) -> tuple[FI, bytes]:
        logger.debug(f'Getting file with {key=}')
        async with self._file_lock.restrict(key), self.context.database.ensure_session(session) as session:
            present_bytes: FI = await self._get_bytes_info(key, session)
        return present_bytes, await self.read(present_bytes.key, byte_range)

    async def read(self, key: str, byte_range: range | None = None) -> bytes:
        logger.debug(f'Reading file with {key=}')
        content = await self._load_from_file_storage(key, byte_range)
        return content

    async def head(self, key: str, session: AnySession = None, nullable: bool = False) -> FI | None:
        logger.info(f'Getting info with {key=}')
        async with self.context.database.ensure_session(session) as session:
            return await self._get_bytes_info(key, session, nullable)

    async def list(self, keys: list[str], session: AnySession = None) -> list[str]:
        logger.info(f'Listing file with specific {len(keys)} keys ')
        async with self.context.database.ensure_session(session) as session:
            infos: list[FI] = await self._get_bytes_infos(keys, session)
        return [info.key for info in infos]

    async def delete(self, key: str, session: AnySession = None) -> FI:
        logger.info(f'Deleting file with key: {key=}')
        async with self._file_lock.restrict(key), self.context.database.ensure_session(session) as session:
            session: WrappedSession
            info: FI = await self.head(key, session=session)
            await self._delete_from_file_storage(info.key)
            if info:
                await session.delete_and_flush(info)
        return info

    def key(self, content: bytes) -> str:
        return self.file_class.from_bytes(content).key
