#  Copyright (C) 2021
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
#
import logging
import sys
from dataclasses import dataclass
from typing import Callable, Any, Type, Union, TypeVar

from .._json import read_json_file_by_path
from .._ini import make_ini_from_hint, read_ini_file_by_path
from .._hints import get_hint_from_dataclass
from .._dataclass import FieldErrors, dict_to_dataclass
from .._exception import repr_exception
from .._toml import read_toml_file_by_path

T = TypeVar('T')
logger = logging.getLogger(__file__)


@dataclass
class Arg:
    caster: Callable[[str], Any] = str
    hint: str = None

    def __call__(self, *args, **kwargs):
        return self.caster(*args, **kwargs)

    def __str__(self):
        return self.hint or 'no description'

    @staticmethod
    def json_file(hint: str = None) -> 'Arg':
        return Arg(read_json_file_by_path, hint)

    @staticmethod
    def ini_file_to_dataclass(dataclass_type: Type[T]) -> 'Arg':
        def ini_file_path_to_dataclass(path: str) -> T:
            return dict_to_dataclass(read_ini_file_by_path(path), dataclass_type)
        return Arg(ini_file_path_to_dataclass, hint=make_ini_from_hint(get_hint_from_dataclass(dataclass_type)))

    @staticmethod
    def toml_file_to_dataclass(dataclass_type: Type[T]) -> 'Arg':
        def toml_file_path_to_dataclass(path: str) -> T:
            return dict_to_dataclass(read_toml_file_by_path(path), dataclass_type)
        return Arg(toml_file_path_to_dataclass, hint=make_ini_from_hint(get_hint_from_dataclass(dataclass_type)))

    @staticmethod
    def ini_file(hint: str = None) -> 'Arg':
        return Arg(read_ini_file_by_path, hint)


def parse_args(args: list[str] | None = None, **kwargs: Union[type, Arg]) -> list:
    if args is None:
        args = sys.argv[1:]

    usage = f'{sys.argv[0]} {" ".join(kwargs)}'
    full_usage = f'{usage}\nArguments:\n'
    for i, key in enumerate(kwargs):
        full_usage += f'{i+1}. {key}:\n{kwargs[key]}\n'

    arg_name_to_exception = {}
    arg_name_result = {}
    for arg_name, caster in kwargs.items():
        if not args:
            arg_name_to_exception[arg_name] = ValueError("missing argument")
            continue
        arg = args.pop(0)
        try:
            arg_name_result[arg_name] = caster(arg)
        except Exception as e:
            arg_name_to_exception[arg_name] = e

    if arg_name_to_exception:
        logger.warning('Usage:'.upper())
        logger.warning(usage)
        logger.critical('Fatal argument errors:'.upper())
        for i, key in enumerate(arg_name_to_exception):
            exception = arg_name_to_exception[key]
            if isinstance(exception, FieldErrors):
                logger.critical(f'{i+1}. {key}:')
                for j, field in enumerate(exception.field_to_error):
                    sub_exception = exception.field_to_error[field]
                    logger.critical(f'{j+1}) {field} - {repr_exception(sub_exception)}')
            else:
                logger.critical(f'{i+1}. {key} - {repr_exception(exception)}')
            logger.warning(f"@{key} hint:")
            logger.warning(kwargs[key])
            logger.warning(f"@{key} hint end")
        # print(exceptions)
        exit(2)
    result = [arg_name_result[key] for key in kwargs]
    if len(result) == 1:
        return result[0]
    return result
