#  Copyright (C) 2022
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasiliev Ivan <i.vasiliev@technokert.ru>
import json
from abc import ABC, abstractmethod
from dataclasses import dataclass
from typing import Optional, Any, Union


@dataclass
class SerializedRecord:
    key: Optional[bytes]
    value: Optional[bytes]


class AbstractSerializer(ABC):

    @abstractmethod
    def serialize(self, *args, **kwargs) -> bytes:
        pass


class StringSerializer(AbstractSerializer):

    def serialize(self,  s: str) -> bytes:
        return s.encode()


class JsonSerializer(StringSerializer):

    def serialize(self, s: Union[dict, str, int]) -> bytes:
        s = json.dumps(s)
        return super().serialize(s)

# TODO consider add IntSerializer


class RecordSerializer:
    @dataclass
    class Context:
        key_serializer: AbstractSerializer
        value_serializer: AbstractSerializer

    def __init__(self, context: Context) -> None:
        self.context = context

    def serialize_record(self, key: Optional[Any], value: Optional[Any]) -> SerializedRecord:
        return SerializedRecord(key=self._serialize_key(key), value=self._serialize_value(value))

    def _serialize_key(self, key: Any) -> Optional[bytes]:
        answer = None
        if key is not None:
            answer = self.context.key_serializer.serialize(key)
        return answer

    def _serialize_value(self, value: Optional[Any]) -> Optional[bytes]:
        answer = None
        if value is not None:
            answer = self.context.value_serializer.serialize(value)
        return answer
