from dataclasses import dataclass, field
from enum import IntEnum

from typing import Optional
from kafka import KafkaAdminClient
from kafka.admin import NewTopic

from ..kafka_client.consumer import Consumer


class KafkaAdminErrorCode(IntEnum):
    topic_not_exist_error_code = 3


class KafkaDictConsumer(Consumer):
    # TODO refactor KafkaDictConsumer to work as single consumer in project
    @dataclass(frozen=True)
    class Config(Consumer.Config):
        topic_auto_offset_is_latest: bool = field(init=False, default=False)
        group_id: None = field(init=False, default=None)
        auto_create_topic_partitions: int = 1
        auto_create_topic_replication_factor: int = 1

    @dataclass
    class Context:
        kafka_admin_client: Optional[KafkaAdminClient] = None

    def __init__(self, config: Config, context: Context) -> None:

        super().__init__(config)
        self.context = context
        if self.context.kafka_admin_client is None:
            self.context.kafka_admin_client = KafkaAdminClient(bootstrap_servers=self.config.address)

    async def _start(self) -> None:
        topics_metadata = self.context.kafka_admin_client.describe_topics(list(self._topic_to_callback.keys()))
        topics_to_create = []
        for metadata in topics_metadata:
            if metadata["error_code"] == KafkaAdminErrorCode.topic_not_exist_error_code:
                topics_to_create.append(NewTopic(metadata["topic"],
                                                 self.config.auto_create_topic_partitions,
                                                 self.config.auto_create_topic_replication_factor,
                                                 topic_configs={"cleanup.policy": "compact"}))
        self.context.kafka_admin_client.create_topics(topics_to_create)
        await super()._start()
