#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasiliev Ivan <i.vasiliev@technokert.ru>

from asyncio import Semaphore
from types import TracebackType
from typing import Type


class SemaphoreWrapper:
    def __init__(self, value: int = 1) -> None:
        self.depth: int = 0
        self._semaphore = Semaphore(value)

    def _increase_depth(self) -> None:
        self.depth += 1

    def _decrease_depth(self) -> None:
        self.depth -= 1

    async def __aenter__(self) -> 'SemaphoreWrapper':
        self._increase_depth()
        await self._semaphore.__aenter__()
        return self

    async def __aexit__(self,
                        exc_type: Type[BaseException] | None,
                        exc: BaseException | None,
                        traceback: TracebackType | None) -> None:
        self._decrease_depth()
        await self._semaphore.__aexit__(exc_type, exc, traceback)

    async def acquire(self) -> None:
        self._increase_depth()
        await self._semaphore.acquire()

    def release(self) -> None:
        self._decrease_depth()
        self._semaphore.release()
