#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

from dataclasses import dataclass

from http_tools import AbmServiceConnector
from init_helpers.dict_to_dataclass import get_converter


class WrongAnswerType(Exception):
    pass


class NetrisProviderConnector:
    Config = AbmServiceConnector.Config

    @dataclass
    class Context(AbmServiceConnector.Context):
        project_name: str

    def __init__(self, config: Config, context: Context) -> None:
        self.config = config
        self.context = context
        self._connector = AbmServiceConnector(config, context)

    async def get_camera_outer_id_to_inner_id(self, outer_ids: list[str]) -> dict[str, int]:
        camera_outer_id_to_inner_id = await self._connector.post(
            '/camera_matching/outer_id_to_inner_id/get', {'ids': outer_ids}, headers=self._prepare_headers(),
        )
        return get_converter(dict[str, int])(camera_outer_id_to_inner_id)

    async def get_camera_inner_id_to_outer_id(self, inner_ids: list[int]) -> dict[int, str]:
        camera_inner_id_to_outer_id = await self._connector.post(
            '/camera_matching/inner_id_to_outer_id/get', {'ids': inner_ids}, headers=self._prepare_headers(),
        )
        return get_converter(dict[int, str])(camera_inner_id_to_outer_id)

    async def get_snapshot_by_outer_camera_id(self, outer_camera_id: str) -> bytes:
        snapshot = await self._connector.get(
            '/snapshot/by_outer_camera_id/get', {"outer_camera_id": outer_camera_id}, headers=self._prepare_headers()
        )

        if not isinstance(snapshot, bytes):
            raise WrongAnswerType(f"expected type bytes, got {type(snapshot)}")

        return snapshot

    async def get_snapshot_by_inner_camera_id(self, inner_camera_id: int) -> bytes:
        snapshot = await self._connector.get(
            '/snapshot/by_inner_camera_id/get', {"inner_camera_id": inner_camera_id}, headers=self._prepare_headers()
        )

        if not isinstance(snapshot, bytes):
            raise WrongAnswerType(f"expected type bytes, got {type(snapshot)}")

        return snapshot

    def _prepare_headers(self) -> dict[str, str]:
        return {"server_name": self.context.project_name}
