import inspect
from inspect import Signature
from typing import Callable, Type

from http_tools import Answer, WrappedAnswer, HttpStatusCode
from http_tools.answer import ExceptionAnswer

from .open_api_server import RpcEndpoint, JsonBodyParameter
from .open_api_wrapper import Security


def gen_json_rpc_endpoint(
        callable_: Callable, securities: list[Security],
        exception_type_to_answer_type: dict[Type[Exception], Type[Answer]],
        unhandled_exception__answer_type: Type[Answer] = ExceptionAnswer[HttpStatusCode.InternalServerError]
) -> RpcEndpoint:
    signature = Signature.from_callable(callable_)
    argument_name_to_parameter = {}
    for param in signature.parameters.values():
        if param.annotation == Signature.empty:
            raise TypeError(f"undefined type for arg: {param}")

        schema = param.annotation
        if param.kind == inspect.Parameter.VAR_POSITIONAL:
            schema = list[schema]
        elif param.kind == inspect.Parameter.VAR_KEYWORD:
            schema = dict[str, schema]

        parameter_kwargs = {"name": param.name, "schema": schema}
        if param.default != Signature.empty:
            parameter_kwargs["default"] = param.default
        argument_name_to_parameter[param.name] = JsonBodyParameter(**parameter_kwargs)

    if signature.return_annotation == Signature.empty:
        raise TypeError("undefined return type")

    unwrapped_answer_type = signature.return_annotation
    return RpcEndpoint(
        callable=callable_,
        securities=securities,
        argument_name_to_parameter=argument_name_to_parameter,
        answer_type=WrappedAnswer[HttpStatusCode.OK][unwrapped_answer_type],
        exception_type_to_answer_type=exception_type_to_answer_type,
        unhandled_exception__answer_type=unhandled_exception__answer_type,
    )
