import inspect
import string
from inspect import Signature
from typing import Callable

from http_tools import Answer, WrappedAnswer, HttpStatusCode
from http_tools.answer import ExceptionAnswer

from .open_api_server import RpcEndpoint
from .security import Security, get_securities_arg_names
from .parameter import SpecParameter, QueryParameter, PathParameter, JsonBodyParameter


def gen_query_rpc_endpoint(
        callable_: Callable, method: str, path: str, securities: list[Security],
        exception_type_to_answer_type: dict[type[Exception], type[ExceptionAnswer]],
        unhandled_exception__answer_type: type[Answer] = ExceptionAnswer[HttpStatusCode.InternalServerError]
) -> RpcEndpoint:
    return _gen_rpc_endpoint(
        QueryParameter, callable_, method, path, securities,
        exception_type_to_answer_type, unhandled_exception__answer_type)


def gen_json_rpc_endpoint(
        callable_: Callable, method: str, path: str, securities: list[Security],
        exception_type_to_answer_type: dict[type[Exception], type[ExceptionAnswer]],
        unhandled_exception__answer_type: type[Answer] = ExceptionAnswer[HttpStatusCode.InternalServerError]
) -> RpcEndpoint:
    return _gen_rpc_endpoint(
        JsonBodyParameter, callable_, method, path, securities,
        exception_type_to_answer_type, unhandled_exception__answer_type)


def _get_path_param_names(path: str) -> set[str]:
    return {i[1] for i in string.Formatter().parse(path) if i[1] is not None}


def _gen_rpc_endpoint(
        param_type: type[SpecParameter], callable_: Callable, method: str, path: str, securities: list[Security],
        exception_type_to_answer_type: dict[type[Exception], type[ExceptionAnswer]],
        unhandled_exception__answer_type: type[Answer] = ExceptionAnswer[HttpStatusCode.InternalServerError]
) -> RpcEndpoint:
    path_param_names = _get_path_param_names(path)
    security_param_names = get_securities_arg_names(securities)
    signature = Signature.from_callable(callable_)
    argument_name_to_parameter = {}
    for param in signature.parameters.values():
        if param.annotation == Signature.empty:
            raise TypeError(f"undefined type for arg: {param}")

        schema = param.annotation
        if param.kind == inspect.Parameter.VAR_POSITIONAL:
            schema = list[schema]
        elif param.kind == inspect.Parameter.VAR_KEYWORD:
            schema = dict[str, schema]

        parameter_kwargs = {"name": param.name, "schema": schema}
        if param.default != Signature.empty:
            parameter_kwargs["default"] = param.default
        if param.name in path_param_names:
            param = PathParameter(**parameter_kwargs)
        if param.name in security_param_names:
            continue
            # param = PathParameter(**parameter_kwargs)
        else:
            # noinspection PyArgumentList
            param = param_type(**parameter_kwargs)
        argument_name_to_parameter[param.name] = param

    if signature.return_annotation == Signature.empty:
        raise TypeError("undefined return type")

    return_type = signature.return_annotation
    return_type = type[None] if return_type is None else return_type

    return RpcEndpoint(
        callable=callable_,
        method=method,
        path=path,
        securities=securities,
        argument_name_to_parameter=argument_name_to_parameter,
        answer_type=WrappedAnswer[HttpStatusCode.OK][return_type],
        exception_type_to_answer_type=exception_type_to_answer_type,
        unhandled_exception__answer_type=unhandled_exception__answer_type,
    )
