#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

import asyncio

from aiohttp import ClientSession, ClientResponse
from aiohttp.typedefs import StrOrURL
from http_tools import HttpStatusCode

from ..collectors.outgoing_http_requests_metrics_collector import OutgoingHttpRequestsMetricsCollector
from ..tools.timer import Timer


class MonitoredClientSession(ClientSession):
    def __init__(self, metrics_collector: OutgoingHttpRequestsMetricsCollector, *args, **kwargs) -> None:
        self._metrics_collector = metrics_collector
        super().__init__(*args, **kwargs)

    async def _request(self, method: str, str_or_url: StrOrURL, **kwargs) -> ClientResponse:
        timer = Timer()
        labels = {self._metrics_collector.Label.method: method.upper(), self._metrics_collector.Label.url: str_or_url}

        self._metrics_collector.outgoing_http_requests__amount__counter.labels(**labels).inc()

        status_code = HttpStatusCode.ServiceUnavailable.value
        output_network_traffic = None
        try:
            with (timer,
                  self._metrics_collector.outgoing_http_requests__current__gauge.labels(**labels).track_inprogress(),
                  ):
                response = await super()._request(method, str_or_url, **kwargs)
            status_code = response.status
            output_network_traffic = response.content_length
            return response
        except asyncio.TimeoutError as er:
            status_code = HttpStatusCode.GatewayTimeout.value
            raise er
        finally:
            labels[self._metrics_collector.Label.status_code] = status_code
            self._metrics_collector.outgoing_http_requests__latency__histogram.labels(**labels).observe(timer.duration)
            if output_network_traffic:
                self._metrics_collector.outgoing_http_requests__input_network_traffic__counter.labels(**labels).inc(
                    output_network_traffic
                )
