#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

from dataclasses import dataclass, field
from typing import List

from prometheus_client import CollectorRegistry, REGISTRY, Counter, Gauge, Histogram
from prometheus_client.utils import INF


class OutgoingHttpRequestsMetricsCollector:
    class Label:
        method = "method"
        url = "url"
        status_code = "status_code"

    @dataclass
    class Context:
        registry: CollectorRegistry = REGISTRY

    @dataclass
    class Config:
        latency_buckets: List[float] = field(default_factory=lambda: [.1, .25, .5, .75, 1.0, 2.5, 5, 7.5, 10.0, INF])

    def __init__(self, config: Config, context: Context) -> None:
        self.config = config
        self.context = context

        self.outgoing_http_requests__amount__counter = Counter(
            "outgoing_http_requests__amount",
            "Total number of outgoing HTTP requests",
            labelnames=[self.Label.method, self.Label.url],
            registry=self.context.registry,
        )

        self.outgoing_http_requests__current__gauge = Gauge(
            "outgoing_http_requests__current",
            "Number of current outgoing HTTP requests",
            labelnames=[self.Label.method, self.Label.url],
            registry=self.context.registry
        )

        self.outgoing_http_requests__latency__histogram = Histogram(
            "outgoing_http_requests__latency",
            "Outgoing HTTP request latency",
            labelnames=[self.Label.method, self.Label.url, self.Label.status_code],
            registry=self.context.registry,
            buckets=self.config.latency_buckets
        )

        self.outgoing_http_requests__input_network_traffic__counter = Counter(
            "outgoing_http_requests__input_network_traffic",
            "Input network traffic of outgoing HTTP requests in bytes",
            labelnames=[self.Label.method, self.Label.url, self.Label.status_code],
            registry=self.context.registry,
        )
        #
        # self.outgoing_http_requests__output_network_traffic__counter = Counter(
        #     "outgoing_http_requests__output_network_traffic",
        #     "Output network traffic of outgoing HTTP requests in bytes",
        #     labelnames=[self.Label.method, self.Label.url, self.Label.status_code],
        #     registry=self.context.registry,
        # )
