#  Copyright (C) 2022
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>
import logging
from contextlib import contextmanager
from dataclasses import dataclass, field

from typing import List, Iterator

from prometheus_client.utils import INF

from ..metrics_registry import METRICS_REGISTRY, MetricsRegistry
from ..tools.timer import Timer

logger = logging.getLogger(__name__)


class BackgroundTasksMetricsCollector:
    class Label:
        name = "name"
        done = "done"

    @dataclass
    class Context:
        registry: MetricsRegistry = METRICS_REGISTRY

    @dataclass
    class Config:
        duration_buckets: List[float] = field(default_factory=lambda: [.1, .5, 1.0, 5, 10.0, 20.0, 30.0, 60.0, INF])

    def __init__(self, config: Config, context: Context):
        self.config = config
        self.context = context

        self.total_background_tasks_counter = self.context.registry.ensure_counter(
            "total_background_tasks",
            "Total number of running background tasks for all time",
            labelnames=[self.Label.name, self.Label.done],
        )

        self.active_background_tasks_gauge = self.context.registry.ensure_gauge(
            "active_background_tasks",
            "Number of active background tasks",
            labelnames=[self.Label.name],
        )

        self.background_tasks_duration_histogram = self.context.registry.ensure_histogram(
            "background_tasks_duration",
            "Background tasks duration",
            labelnames=[self.Label.name, self.Label.done],
            buckets=self.config.duration_buckets
        )

    @contextmanager
    def track(self, task_name: str) -> Iterator[None]:
        labels = {self.Label.name: task_name}
        timer = Timer()
        try:
            with timer, self.active_background_tasks_gauge.labels(**labels).track_inprogress():
                yield
        except Exception as er:
            labels[self.Label.done] = False
            raise er
        else:
            labels[self.Label.done] = True
        finally:
            self.background_tasks_duration_histogram.labels(**labels).observe(timer.duration)
            self.total_background_tasks_counter.labels(**labels).inc()
