#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>

from dataclasses import dataclass, field
from typing import List

from prometheus_client.utils import INF

from ..metrics_registry import MetricsRegistry, METRICS_REGISTRY


class OutgoingHttpRequestsMetricsCollector:
    class Label:
        method = "method"
        url = "url"
        status_code = "status_code"

    @dataclass
    class Context:
        registry: MetricsRegistry = METRICS_REGISTRY

    @dataclass
    class Config:
        latency_buckets: List[float] = field(default_factory=lambda: [.1, .25, .5, .75, 1.0, 2.5, 5, 7.5, 10.0, INF])

    def __init__(self, config: Config, context: Context) -> None:
        self.config = config
        self.context = context

        self.outgoing_http_requests__amount__counter = self.context.registry.ensure_counter(
            "outgoing_http_requests__amount",
            "Total number of outgoing HTTP requests",
            labelnames=[self.Label.method, self.Label.url],
        )

        self.outgoing_http_requests__current__gauge = self.context.registry.ensure_gauge(
            "outgoing_http_requests__current",
            "Number of current outgoing HTTP requests",
            labelnames=[self.Label.method, self.Label.url],
        )

        self.outgoing_http_requests__latency__histogram = self.context.registry.ensure_histogram(
            "outgoing_http_requests__latency",
            "Outgoing HTTP request latency",
            labelnames=[self.Label.method, self.Label.url, self.Label.status_code],
            buckets=self.config.latency_buckets
        )

        self.outgoing_http_requests__input_network_traffic__counter = self.context.registry.ensure_counter(
            "outgoing_http_requests__input_network_traffic",
            "Input network traffic of outgoing HTTP requests in bytes",
            labelnames=[self.Label.method, self.Label.url, self.Label.status_code],
        )
        #
        # self.outgoing_http_requests__output_network_traffic__counter = Counter(
        #     "outgoing_http_requests__output_network_traffic",
        #     "Output network traffic of outgoing HTTP requests in bytes",
        #     labelnames=[self.Label.method, self.Label.url, self.Label.status_code],
        #     registry=self.context.registry,
        # )
