#  Copyright (C) 2022
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Alexander Medvedev <a.medvedev@abm-jsc.ru>

import logging
from abc import ABC, abstractmethod
from dataclasses import dataclass
from typing import NoReturn, Optional

from aio_pika import IncomingMessage
from aio_pika.abc import AbstractRobustChannel, AbstractExchange


logger = logging.getLogger(__name__)


class RabbitMQConsumer(ABC):
    @dataclass
    class Config:
        queue_name: str = None
        durable: bool = False
        exclusive: bool = False
        passive: bool = False
        auto_delete: bool = False
        arguments: dict = None
        timeout: float = None
        robust: bool = True

    def __init__(self, config: Config):
        self._config = config

    @abstractmethod
    async def _process(self, msg: bytes) -> None:
        pass

    async def run(self, channel: AbstractRobustChannel, exchange: Optional[AbstractExchange] = None) -> NoReturn:
        queue = await channel.declare_queue(
            name=self._config.queue_name, durable=self._config.durable,
            exclusive=self._config.exclusive, passive=self._config.passive,
            auto_delete=self._config.auto_delete, arguments=self._config.arguments,
            timeout=self._config.timeout, robust=self._config.robust
        )
        if exchange is not None:
            await queue.bind(exchange)

        async with queue.iterator() as iterator:
            message: IncomingMessage
            async for message in iterator:
                try:
                    await self._process(message.body)
                    await message.ack()
                    logger.info(
                        f'Message with id "{message.message_id}" from queue "{self._config.queue_name}" has been acked'
                    )
                except Exception as err:
                    logger.error(
                        f'Failed to process error at the queue: "{self._config.queue_name}". Error: {repr(err)}'
                    )
                    raise err
