#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Ruslan Albakov <r.albakov@abm-jsc.ru>
import logging
from dataclasses import dataclass

import redis.asyncio as redis
from async_tools import AsyncInitable, AsyncDeinitable

logger = logging.getLogger(__name__)

class RedisConnector(AsyncInitable, AsyncDeinitable):
    @dataclass
    class Config:
        url: str = 'redis://localhost:6379/0'

    async def _async_init(self) -> None:
        self._connection = await redis.from_url(self.config.url)

    async def _async_deinit(self) -> None:
        if self._connection:
            # to prevent possibility of multiple "close" calls from different contexts
            conn, self._connection = self._connection, None
            await conn.close()

    def __init__(self, config: Config):
        AsyncInitable.__init__(self)
        AsyncDeinitable.__init__(self)
        self.config = config
        self._connection: redis.Redis | None = None

    async def hdel(self, hash_name: str, *keys: str) -> None:
        """
        Delete ``keys`` from hash ``hash_name``.
        :return: None.
        """
        logger.debug(f"Delete {keys}")
        await self._connection.hdel(hash_name, *keys)

    async def hset_and_get_hlen(self,
                                hash_name: str,
                                key: str,
                                value: str = '',
                                seconds: int = 120) -> int:
        """
        Set ``key`` within hash ``hash_name``, set the expiration time for key in seconds \
        and return the number of elements in hash ``hash_name``
        :param hash_name: str, the name of the hash.
        :param key: str, set field in the hash stored at key.
        :param value: str, default ''.
        :param seconds: int, set expiration time in seconds.
        :return: int, number of elements in hash ``hash_name``.
        """
        async with self._connection.pipeline(transaction=True) as pipeline:
            await pipeline.hset(name=hash_name, key=key, value=value)
            await pipeline.hexpire(hash_name, seconds, key, nx=True)
            await pipeline.hlen(hash_name)
            *_, hlen = await pipeline.execute()
            logger.debug(f'Added key. Hash len = {hlen}')
            return hlen
