#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
import json

from run_markdown.base_comparator import BaseComparator
from run_markdown.comparison_buffer import ComparisonBuffer
from run_markdown.comparison_line import JsonComparisonLine
from run_markdown.utils import Missing


class JsonComparator(BaseComparator):
    def compare(self, value: str) -> ComparisonBuffer:
        result = ComparisonBuffer()
        expected = actual = Missing
        try:
            expected = json.loads(self.expected)
        except json.decoder.JSONDecodeError:
            result.append(JsonComparisonLine(self.expected, '', ValueError("JSON parse failed")))
        try:
            actual = json.loads(value)
        except json.decoder.JSONDecodeError:
            result.append(JsonComparisonLine('', value, ValueError("JSON parse failed")))
        if expected == Missing or actual == Missing:
            return result

        self._compare(expected, actual, result)
        return result

    def _compare(
            self,
            expected: dict | list | str | float | int | None,
            actual: dict | list | str | float | int | None | type[Missing],
            buffer: ComparisonBuffer,
            comma: bool = False,
            prefix: str = ''
    ) -> None:
        comma_str = ',' if comma else ''
        expected_str = prefix + json.dumps(expected) + comma_str
        actual_str = prefix + json.dumps(actual) + comma_str
        if type(expected) is not type(actual):
            buffer.append(JsonComparisonLine(expected_str, actual_str, ValueError(
                f"Expected {type(expected).__name__}, got {type(actual).__name__}")))
        elif isinstance(expected, str | float | int):
            result = None if expected == actual else ValueError(f"Expected {expected}, got {actual}")
            buffer.append(JsonComparisonLine(expected_str, actual_str, result))
        elif isinstance(expected, list):
            buffer.append(JsonComparisonLine(prefix + '[', prefix + '['))
            buffer.append(sub_buffer := ComparisonBuffer([]))
            for expected_val, actual_val in zip(expected, actual):
                self._compare(expected_val, actual_val, sub_buffer, comma=True)
            for i in range(len(actual), len(expected)):
                sub_buffer.append(JsonComparisonLine(f'{expected[i]},', '', ValueError(f"Missing item={expected[i]}")))
            for i in range(len(expected), len(actual)):
                sub_buffer.append(JsonComparisonLine('', f'{actual[i]},', ValueError(f"Extra item={actual[i]}")))
            buffer.append(JsonComparisonLine(']' + comma_str, ']' + comma_str))
        elif isinstance(expected, dict):
            buffer.append(JsonComparisonLine(prefix + '{', prefix + '{'))
            buffer.append(sub_buffer := ComparisonBuffer([]))
            for key, expected_val in expected.items():
                sub_prefix = json.dumps(key) + ": "
                actual_val = actual.get(key, Missing)
                if actual_val is Missing:
                    sub_buffer.append(JsonComparisonLine(
                        f'{sub_prefix}{expected_val},', '', ValueError(f"Missing {key=!r}")))
                else:
                    self._compare(expected_val, actual_val, sub_buffer, prefix=sub_prefix, comma=True)
            for key in set(actual.keys()) - set(expected.keys()):
                sub_prefix = json.dumps(key) + ": "
                sub_buffer.append(JsonComparisonLine(
                    '', f'{sub_prefix}{actual[key]},', ValueError(f"Unexpected {key=}")))
            buffer.append(JsonComparisonLine('}' + comma_str, '}' + comma_str))
