#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Revva Konstantin <k.revva@abm-jsc.ru>


from dataclasses import dataclass
from typing import Type

from http_tools.http_server_connector import HttpServerConnector
from init_helpers.dict_to_dataclass import dict_to_dataclass, T
from init_helpers.custom_json import custom_dumps

from detector_connector.detector_kafka_connector import DetectionAnalyzeResultType

DetectorSettingsType = Type[T]


class DetectorHttpConnector:
    @dataclass
    class Context(HttpServerConnector.Context):
        project_name: str

    Config = HttpServerConnector.Config

    def __init__(self,
                 config: Config,
                 context: Context,
                 detector_settings_type: DetectorSettingsType) -> None:
        self.config = config
        self.context = context
        self._connector = HttpServerConnector(config, context)
        self.detector_settings_type = detector_settings_type

    async def get_settings(self, stream_url: str, token: str) -> T:
        payload = {
                "token": token,
                "stream_url": stream_url
        }
        answer = await self._connector.get("/settings/get",
                                           url_query=payload,
                                           headers=self._construct_headers())

        return dict_to_dataclass(answer["done"], self.detector_settings_type)

    async def set_settings(self, detector_settings: DetectorSettingsType, token: str) -> T:
        payload = {"token": token}
        payload.update(custom_dumps(detector_settings))
        answer = await self._connector.post_json("/settings/correct",
                                                 payload=payload,
                                                 headers=self._construct_headers())
        return dict_to_dataclass(answer["done"], self.detector_settings_type)

    async def autotune_settings(self, detection: DetectionAnalyzeResultType, token: str) -> T:
        payload = {"token": token}
        payload.update({"detection": custom_dumps(detection)})
        answer = await self._connector.post_json("/settings/correct",
                                                 payload=payload,
                                                 headers=self._construct_headers())
        return dict_to_dataclass(answer["done"], self.detector_settings_type)

    def _construct_headers(self) -> dict[str, str]:
        return {"server_name": self.context.project_name}
