#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Revva Konstantin <k.revva@abm-jsc.ru>


from dataclasses import dataclass
from typing import Type, Any
import json

from http_tools.http_server_connector import HttpServerConnector
from init_helpers.dict_to_dataclass import dict_to_dataclass, T
from init_helpers.custom_json import custom_dumps

DetectorSettingsType = Type[T]


class DetectorHttpConnector:
    @dataclass
    class Context(HttpServerConnector.Context):
        project_name: str

    Config = HttpServerConnector.Config

    def __init__(self,
                 config: Config,
                 context: Context,
                 detector_settings_type: DetectorSettingsType) -> None:
        self.config = config
        self.context = context
        self._connector = HttpServerConnector(config, context)
        self.detector_settings_type = detector_settings_type

    async def get_settings(self, stream_urls: list[str], token: str) -> list[T]:
        payload = {
                "token": token,
                "stream_url": stream_urls
        }
        answer = await self._connector.get("/settings/get",
                                           url_query=payload,
                                           headers=self._construct_headers())

        return [dict_to_dataclass(settings, self.detector_settings_type) for settings in answer["result"]]

    async def set_settings(self, detector_settings: T, token: str) -> T:
        payload = {"token": token}
        payload.update(json.loads(custom_dumps(detector_settings)))
        answer = await self._connector.post_json("/settings/set",
                                                 payload=payload,
                                                 headers=self._construct_headers())
        return dict_to_dataclass(answer["result"], self.detector_settings_type)

    async def autotune_settings(self, detection: T, token: str) -> T:
        payload = {"token": token}
        payload.update({"detection": json.loads(custom_dumps(detection))})
        answer = await self._connector.post_json("/settings/autotune",
                                                 payload=payload,
                                                 headers=self._construct_headers())
        return dict_to_dataclass(answer["result"], self.detector_settings_type)

    def _construct_headers(self) -> dict[str, str]:
        return {"server_name": self.context.project_name}
