#  Copyright (C) 2025
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Aleksandr Medvedev <a.medvedev@abm-jsc.ru>

from dataclasses import dataclass

from http_tools import AbmServiceConnector
from init_helpers.dict_to_dataclass import dict_to_dataclass
from team_member_server_connector.action_code import ActionCode


@dataclass(frozen=True, slots=True)
class Scopes:
    course_ids: tuple[int, ...] | None = None
    community_ids: tuple[int, ...] | None = None


@dataclass(frozen=True, slots=True)
class TeamMemberRole:
    id: int
    title: str
    action_codes: frozenset[ActionCode]
    scopes: Scopes


@dataclass(frozen=True, slots=True)
class TeamMemberAccess:
    portal_member_id: int
    team_member_role: TeamMemberRole


class TeamMemberServerConnector:
    Config = AbmServiceConnector.Config

    @dataclass(slots=True)
    class Context(AbmServiceConnector.Context):
        server_name: str

    def __init__(self, config: Config, context: Context) -> None:
        self._context = context
        self._http_connector = AbmServiceConnector(config, context)

    async def get_team_member_access(self, portal_member_id: int) -> TeamMemberAccess:
        answer = await self._http_connector.post(
            "/v1/team_member/access/get",
            payload={"portal_member_id": portal_member_id},
            headers={"server_name": self._context.server_name},
        )
        return dict_to_dataclass(answer, TeamMemberAccess)
