#  Copyright (C) 2022
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Vasya Svintsov <v.svintsov@techokert.ru>, Alexandr Medvedev <a.medvedev@abm-jsc.ru>

from dataclasses import dataclass
from typing import Optional, List, Dict, Any

from .defs import TicketType, TicketListResponse
from .tech_support_connector import TechSupportConnector


class TechSupportTicketsFacade:
    @dataclass
    class Context:
        tech_support_connector: TechSupportConnector

    def __init__(self, context: Context):
        self.context = context

    async def load(self,
                   fields: Optional[List[str]] = None,
                   filters: Optional[Dict[str, Any]] = None,
                   limit: int = 50,
                   since: Optional[str] = None
                   ) -> TicketListResponse:
        response = await self.context.tech_support_connector.get_ticket_list(fields, filters, limit, since)
        return response

    async def load_one_or_none(self,
                               fields: Optional[List[str]] = None,
                               filters: Optional[Dict[str, Any]] = None,
                               ) -> Optional[Dict[str, Any]]:
        response = await self.load(fields, filters, limit=1)
        return response.tickets[0] if response.tickets else None

    async def load_all(self,
                       fields: Optional[List[str]] = None,
                       filters: Optional[Dict[str, Any]] = None,
                       ) -> List[Dict[str, Any]]:
        result = []
        since = None
        while True:
            response = await self.context.tech_support_connector.get_ticket_list(fields, filters, since=since)
            since = response.since
            result += response.tickets
            if not since:
                break
        return result

    async def add(self,
                  values_to_create: Dict[str, Any]
                  ) -> Dict[str, Any]:
        return await self.context.tech_support_connector.create_ticket(values_to_create)

    async def info(self,
                   uuid: str,
                   type_: TicketType
                   ) -> Dict[str, Any]:
        return await self.context.tech_support_connector.get_ticket(uuid, type_)

    async def update(self,
                     uuid: str,
                     type_: TicketType,
                     values_to_update: Dict[str, Any]
                     ) -> Dict[str, Any]:
        return await self.context.tech_support_connector.update_ticket(uuid, type_, values_to_update)

    async def load_stages(self,
                          uuid: str,
                          type_: TicketType
                          ) -> Dict[str, Any]:
        return await self.context.tech_support_connector.get_ticket_stages(uuid, type_)
