#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
import datetime
import logging
import time
from typing import overload, ClassVar

logger = logging.getLogger(__name__)
MILLISECONDS_PER_SECOND = 1000


class TimestampSec(float):
    _MAX_TIMESTAMP_SEC: ClassVar[int] = 2 ** 32
    DEFAULT_DATETIME_FORMAT: ClassVar[str] = '%Y-%m-%d %H:%M:%S'
    DEFAULT_DATE_FORMAT: ClassVar[str] = '%Y-%m-%d'

    def __new__(cls, value: int | float | datetime.datetime) -> 'TimestampSec':
        if isinstance(value, datetime.datetime):
            if value.tzinfo is None:
                raise ValueError("Attempt to make Timestamp from naive datetime"
                                 "(https://docs.python.org/3/library/datetime.html#aware-and-naive-objects)")
            value = value.timestamp()

        if not isinstance(value, int | float) or isinstance(value, bool):
            raise TypeError(f"Can not convert {type(value).__name__} to {cls.__name__}: {value!r}")

        # value = float(value)  # to cast possible string values and raise default exception
        while abs(value) > cls._MAX_TIMESTAMP_SEC:
            value /= MILLISECONDS_PER_SECOND

        # noinspection PyArgumentList
        return float.__new__(cls, value)

    def to_datetime(self, timezone: datetime.timezone = None) -> datetime.datetime:
        timezone = datetime.timezone.utc if timezone is None else timezone
        return datetime.datetime.fromtimestamp(self, tz=timezone)

    @staticmethod
    def now() -> 'TimestampSec':
        return TimestampSec(time.time())

    @classmethod
    def from_iso_str(cls, string_value: str, default_timezone: datetime.timezone | None = None) -> 'TimestampSec':
        datetime_ = datetime.datetime.fromisoformat(string_value)
        if datetime_.tzinfo is None:
            if default_timezone is None:
                raise ValueError("Timezone in required, but no timezone specified in value and no default passed")
            datetime_ = datetime_.replace(tzinfo=default_timezone)
        return TimestampSec(datetime_.timestamp())

    @classmethod
    def from_datetime_str(
            cls, string_value: str,
            timezone: datetime.timezone | None = None,
            format_: str | None = None,
    ) -> 'TimestampSec':
        format_ = format_ or cls.DEFAULT_DATETIME_FORMAT
        if "%z" in format_:
            if timezone is None:
                dt = datetime.datetime.strptime(string_value, format_)
            else:
                raise ValueError("timezone is defined both in datetime_format and passed as argument")
        else:
            if timezone is None:
                raise ValueError("timezone is required to make timestamp from datetime str, "
                                 "but is not contained in 'datetime_format', nor passed as 'timezone' argument")
            else:
                dt = datetime.datetime.strptime(string_value, format_)
                dt = dt.replace(tzinfo=timezone)
        return TimestampSec(dt.timestamp())

    @classmethod
    def from_date_str(
            cls, string_value: str,
            timezone: datetime.timezone | None = None,
            format_: str | None = None,
    ) -> 'TimestampSec':
        format_ = format_ or cls.DEFAULT_DATE_FORMAT
        return cls.from_datetime_str(string_value=string_value, timezone=timezone, format_=format_)

    def to_datetime_str(self, timezone: datetime.timezone = None, format_: str | None = None) -> str:
        format_ = format_ or self.DEFAULT_DATETIME_FORMAT
        return self.to_datetime(timezone=timezone).strftime(format_)

    def to_date_str(self, timezone: datetime.timezone = None, format_: str | None = None) -> str:
        format_ = format_ or self.DEFAULT_DATE_FORMAT
        return self.to_datetime(timezone=timezone).strftime(format_)

    @overload
    def __sub__(self, other: datetime.timedelta) -> 'TimestampSec': pass
    @overload
    def __sub__(self, other: 'TimestampSec') -> datetime.timedelta: pass

    def __sub__(self, other: 'datetime.timedelta | TimestampSec') -> 'TimestampSec | datetime.timedelta':
        self_type = type(self)
        if isinstance(other, self_type):
            return datetime.timedelta(seconds=float.__sub__(self, other))
        if isinstance(other, datetime.timedelta):
            return self_type(float.__sub__(self, other.total_seconds()))
        if isinstance(other, int | float):
            logger.warning(f"{self_type.__name__} - int|float is unrecommended, "
                           f"use {self_type.__name__} - {self_type.__name__}|datetime.timedelta")
            return self_type(float.__sub__(self, other))
        raise TypeError(f"unsupported operand type for {self_type.__name__}.__sub__: {type(other).__name__}")

    def __add__(self, other: datetime.timedelta) -> 'TimestampSec':
        self_type = type(self)
        if isinstance(other, self_type):
            cls_name = self_type.__name__
            raise TypeError(f"{cls_name} + {cls_name} is prohibited, use {cls_name} + datetime.timedelta")
        if isinstance(other, datetime.timedelta):
            return self_type(float.__add__(self, other.total_seconds()))
        if isinstance(other, int | float):
            logger.warning(f"{self_type.__name__} + int|float is unrecommended, "
                           f"use {self_type.__name__} + datetime.timedelta")
            return self_type(float.__add__(self, other))
        raise TypeError(f"unsupported operand type for {self_type.__name__}.__add__: {type(other).__name__}")

    def __eq__(self, other) -> bool:
        return isinstance(other, type(self)) and float.__eq__(self, other)

    def __ne__(self, other) -> bool:
        return not isinstance(other, type(self)) or not float.__eq__(self, other)

    def __hash__(self) -> int:
        return float.__hash__(self)
