#  Copyright (C) 2024
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
import abc
import datetime
import logging
import time
from typing import overload, ClassVar, Self, Any

logger = logging.getLogger(__name__)
MILLISECONDS_PER_SECOND = 1000


class Timestamp(abc.ABC):
    _MAX_TIMESTAMP_SEC: ClassVar[int] = 2 ** 32
    DEFAULT_DATETIME_FORMAT: ClassVar[str] = '%Y-%m-%dT%H:%M:%S'
    DEFAULT_DATE_FORMAT: ClassVar[str] = '%Y-%m-%d'

    def __init__(self, value: int | float | datetime.datetime) -> None:
        pass

    @classmethod
    def _prepare_value(cls, value: int | float | datetime.datetime) -> int | float:
        if isinstance(value, datetime.datetime):
            if value.tzinfo is None:
                raise ValueError("Attempt to make Timestamp from naive datetime"
                                 "(https://docs.python.org/3/library/datetime.html#aware-and-naive-objects)")
            value = value.timestamp()

        if not isinstance(value, int | float) or isinstance(value, bool):
            raise TypeError(f"Can not convert {type(value).__name__} to {cls.__name__}: {value!r}")
        return value

    @abc.abstractmethod
    def total_seconds(self) -> float:
        raise NotImplementedError

    def total_milliseconds(self) -> float:
        return self.total_seconds() * MILLISECONDS_PER_SECOND

    def to_datetime(self, timezone: datetime.timezone = None) -> datetime.datetime:
        timezone = datetime.timezone.utc if timezone is None else timezone
        return datetime.datetime.fromtimestamp(self.total_seconds(), tz=timezone)

    @classmethod
    def now(cls) -> Self:
        return cls(time.time())

    @classmethod
    def from_iso_str(cls, string_value: str, default_timezone: datetime.timezone | None = None) -> Self:
        datetime_ = datetime.datetime.fromisoformat(string_value)
        if datetime_.tzinfo is None:
            if default_timezone is None:
                raise ValueError("Timezone in required, but no timezone specified in value and no default passed")
            datetime_ = datetime_.replace(tzinfo=default_timezone)
        return cls(datetime_.timestamp())

    @classmethod
    def from_datetime_str(
            cls, string_value: str,
            timezone: datetime.timezone | None = None,
            format_: str | None = None,
    ) -> Self:
        format_ = format_ or cls.DEFAULT_DATETIME_FORMAT
        if "%z" in format_:
            if timezone is None:
                dt = datetime.datetime.strptime(string_value, format_)
            else:
                raise ValueError("timezone is defined both in datetime_format and passed as argument")
        else:
            if timezone is None:
                raise ValueError("timezone is required to make timestamp from datetime str, "
                                 "but is not contained in 'datetime_format', nor passed as 'timezone' argument")
            else:
                dt = datetime.datetime.strptime(string_value, format_)
                dt = dt.replace(tzinfo=timezone)
        return cls(dt)

    @classmethod
    def from_date_str(
            cls, string_value: str,
            timezone: datetime.timezone | None = None,
            format_: str | None = None,
    ) -> Self:
        format_ = format_ or cls.DEFAULT_DATE_FORMAT
        return cls.from_datetime_str(string_value=string_value, timezone=timezone, format_=format_)

    def to_datetime_str(self, timezone: datetime.timezone = None, format_: str | None = None) -> str:
        format_ = format_ or self.DEFAULT_DATETIME_FORMAT
        return self.to_datetime(timezone=timezone).strftime(format_)

    def to_date_str(self, timezone: datetime.timezone = None, format_: str | None = None) -> str:
        format_ = format_ or self.DEFAULT_DATE_FORMAT
        return self.to_datetime(timezone=timezone).strftime(format_)

    @overload
    def __sub__(self, other: datetime.timedelta) -> Self: pass
    @overload
    def __sub__(self, other: Self) -> datetime.timedelta: pass

    def __sub__(self, other: datetime.timedelta | Self) -> Self | datetime.timedelta:
        self_type = type(self)
        if isinstance(other, datetime.timedelta):
            return self_type(self.total_seconds() - other.total_seconds())
        if isinstance(other, self_type):
            return datetime.timedelta(seconds=self.total_seconds() - other.total_seconds())
        cls_name = self_type.__name__
        if isinstance(other, int | float):
            raise TypeError(f"{cls_name} - int|float is prohibited, use {cls_name} - {cls_name}|datetime.timedelta")
        raise TypeError(f"unsupported operand type for {cls_name}.__sub__: {type(other).__name__}")

    def __add__(self, other: datetime.timedelta) -> Self:
        self_type = type(self)
        if isinstance(other, datetime.timedelta):
            return self_type(self.total_seconds() + other.total_seconds())
        cls_name = self_type.__name__
        if isinstance(other, self_type):
            raise TypeError(f"{cls_name} + {cls_name} is prohibited, use {cls_name} + datetime.timedelta")
        if isinstance(other, int | float):
            raise TypeError(f"{cls_name} + int|float is prohibited, use {cls_name} + datetime.timedelta")
        raise TypeError(f"unsupported operand type for {cls_name}.__add__: {type(other).__name__}")

    def __eq__(self, other: Any) -> bool:
        return isinstance(other, type(self)) and self.total_seconds() == other.total_seconds()

    def __ne__(self, other: Any) -> bool:
        return not isinstance(other, type(self)) or self.total_seconds() != other.total_seconds()

    def __hash__(self) -> int:
        return float.__hash__(self.total_seconds())
