#  Copyright (C) 2023
#  ABM, Moscow
#
#  UNPUBLISHED PROPRIETARY MATERIAL.
#  ALL RIGHTS RESERVED.
#
#  Authors: Mike Orlov <m.orlov@abm-jsc.ru>
import dataclasses
import types
from typing import Any, Type, Callable

from init_helpers import is_instance
from init_helpers.dict_to_dataclass import get_dataclass_field_name_to_field, NoValue

from .defs import Json
from .typed_In_dumps import TypedInDumps

LiteralTypeWrapper = Callable[[dict | list | str | float | int | bool | None], Any]


class TypedJsonParser:
    def __init__(self, literal_type_wrapper: LiteralTypeWrapper = lambda x: x):
        self.literal_type_wrapper = literal_type_wrapper

    def parse(self, value: Any, target_type: Type[TypedInDumps | Json] | types.UnionType = TypedInDumps):
        if is_instance(value, [target_type]):
            return value

        if target_type is self.literal_type_wrapper:
            if not isinstance(value, TypedInDumps):
                value = self.parse(
                    value, dict[str, TypedInDumps] | list[TypedInDumps] | str | float | int | bool | None)
            return self.literal_type_wrapper(value)

        if isinstance(target_type, type) and issubclass(target_type, TypedInDumps):
            if isinstance(value, target_type):
                return value
            if isinstance(value, dict) and '_t' in value:
                type_name = value.pop('_t')
                prefixed_name_to_non_abstract_type = target_type.get_prefixed_name_to_non_abstract_subclass()
                try:
                    type_ = prefixed_name_to_non_abstract_type[type_name]  # TODO add some kind of skip for unknown
                except KeyError as e:
                    raise KeyError(
                        f"required {target_type.__name__}({list(prefixed_name_to_non_abstract_type)}), "
                        f"got: {type_name}") from e

                kwargs = {}
                for key, field in get_dataclass_field_name_to_field(type_).items():
                    val = value.get(key, NoValue) if field.init else NoValue
                    if val is NoValue:
                        if dataclasses.is_dataclass(field.type):
                            val = self.parse({}, field.type)
                    else:
                        val = self.parse(val, field.type)
                    if val is not NoValue:
                        kwargs[key] = val
                return type_(**kwargs)
            return self.parse(value, self.literal_type_wrapper)

        target_type_origin = getattr(target_type, '__origin__', None)
        target_type_args = getattr(target_type, '__args__', None)

        if isinstance(target_type, types.UnionType):
            if is_instance(value, target_type_args):
                return value
            for possible_type in target_type_args:
                try:
                    return self.parse(value, possible_type)
                except (ValueError, TypeError, AssertionError):
                    continue
            raise ValueError(f"failed all cast attempts: {value} -> {target_type}")

        if target_type_origin is list:
            assert isinstance(value, list)
            assert len(target_type_args) == 1
            return [self.parse(val, target_type_args[0]) for val in value]

        if target_type_origin is dict:
            assert isinstance(value, dict)
            assert len(target_type_args) == 2
            key_type, value_type = target_type_args
            return {self.parse(key, key_type): self.parse(val, value_type) for key, val in value.items()}

        if target_type in (str, bool) and not isinstance(value, target_type):
            raise ValueError(f"Attempt to cast {value} as {target_type}")
        return target_type(value)
